/**
* \file: Factory.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Utility
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef UTILITY_FACTORY_H
#define UTILITY_FACTORY_H

#include <map>
#include <string>
#include <typeinfo>

#include "Singleton.h"

namespace adit { namespace utility {

class Factory
{
// not thread-safe
public:
    typedef void* (*CreateImplFn)(void);
    typedef void (*LogFn)(const char*);

    Factory();

    // create an implementation based on IFType interface (works only with default constructor)
    template<class IFType>
    IFType* create(const std::string& inName)
    {
        auto impl = _getImpl(inName, typeid(IFType), typeid(CreateImplFn));
        if (impl == nullptr)
            return nullptr; // error is already logged
        return static_cast<IFType*>(impl->creatorFunction());
    };

    // return the creator function of an implementation, useful for parameterized constructors
    template<class IFType, typename ConstructorType = CreateImplFn>
    ConstructorType getCreator(const std::string& inName)
    {
        auto impl = _getImpl(inName, typeid(IFType), typeid(ConstructorType));
        if (impl == nullptr)
            return nullptr; // error is already logged
        return reinterpret_cast<ConstructorType>(impl->creatorFunction);
    }

    // register an implementation based on IFType interface
    template<class IFType, typename ConstructorType = CreateImplFn>
    bool registerImpl(const std::string& inName, ConstructorType inCreator)
    {
        return _register(inName, typeid(IFType), typeid(ConstructorType),
                reinterpret_cast<CreateImplFn>(inCreator));
    };
    void setLogger(LogFn inDebugLevel, LogFn inErrorLevel);

    // helper template creator method
    template<class ImplType>
    static void* creator() { return new ImplType(); }

private:
    struct Impl
    {
        const std::type_info& typeInfo;
        const std::type_info& constructorInfo;
        CreateImplFn creatorFunction;
    };
    std::map<std::string, Impl> factoryMap;
    LogFn debugLevel;
    LogFn errorLevel;

    //void* _create(const std::string& inName, const std::type_info& inTypeInfo);

    Impl* _getImpl(const std::string& inName, const std::type_info& inTypeInfo,
            const std::type_info& inConstructorInfo);
    bool _register(const std::string& inName, const std::type_info& inTypeInfo,
            const std::type_info& inConstructorInfo, CreateImplFn inCreator);

    static void logMsg(LogFn inLevel, const char* msg, ...);
};

} } /* namespace adit { namespace utility */

#endif /* UTILITY_FACTORY_H */
